//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// X68k キーボード
//

#pragma once

#include "keyboard.h"
#include <array>

class MFPDevice;
class SCCDevice;
class SysportDevice;

class X68030Keyboard : public Keyboard
{
	using inherited = Keyboard;

	// マウスステータスデータ
	static const uint8 YUNFL	= 0x80;
	static const uint8 YOVFL	= 0x40;
	static const uint8 XUNFL	= 0x20;
	static const uint8 XOVFL	= 0x10;
	static const uint8 SW_R		= 0x02;
	static const uint8 SW_L		= 0x01;

 public:
	X68030Keyboard();
	~X68030Keyboard() override;

	bool Init() override;

	// シフト状態なら true を返す
	bool IsShift() const override;

	// マウス入力
	void MouseInput(int x, int y, bool rb, bool lb, bool mb) override;

	// マウスデータを VM に送信する MD 処理
	void MouseSendStart() override;

	// ホストからの制御コマンド
	void Command(uint32 data) override;

	// ホストからの Ready 通知
	// (実際のハードウェアでは単にキーボードが待っている)
	void Ready();

 private:
	DECLARE_MONITOR_SCREEN(MonitorScreen);

	// キーボードの接続と取り外し
	void MDConnect() override;
	void MDDisconnect() override;

	// MD キーコードに変換
	uint KeyToMDKey(uint keystat) const override;

	// キーを1つ処理する
	void KeyInput(uint keystat) override;

	// キーイベントを登録(追加)する
	void SendStart() override;

	// キーリピートイベント
	void ReptCallback(Event *);

	// マウスイベント
	void MouseCallback(Event *);

	// 変換テーブル
	static const uint keycode2x68kkey_table[KC_max];

	// キーリピート開始時間
	uint rept_delay_data {};	// 設定値
	uint64 rept_delay {};		// [nsec]
	// キーリピート間隔
	uint rept_time_data {};		// 設定値
	uint64 rept_time {};		// [nsec]
	// キーリピートイベント
	Event *rept_event {};
	// リピートしている共通キーコード
	uint rept_key {};

	// LED 暗さ (0 明るい <-> 3 暗い)
	uint32 led_darkness {};

	// LED は以下の7つ。
	// 0: かな
	// 1: ローマ字
	// 2: コード入力
	// 3: CAPS
	// 4: INS
	// 5: ひらがな
	// 6: 全角

	// マウス送信データ
	std::array<uint8, 3> mouse_data {};

	MFPDevice *mfp {};
	SCCDevice *scc {};
	SysportDevice *sysport {};

	// マウスイベント
	Event *mouse_event {};

	Monitor *monitor {};

	// X680x0 キーコード
	static const uint NoKey					= 0;
	static const uint X68kKey_ESC			= 0x01;
	static const uint X68kKey_1				= 0x02;
	static const uint X68kKey_2				= 0x03;
	static const uint X68kKey_3				= 0x04;
	static const uint X68kKey_4				= 0x05;
	static const uint X68kKey_5				= 0x06;
	static const uint X68kKey_6				= 0x07;
	static const uint X68kKey_7				= 0x08;
	static const uint X68kKey_8				= 0x09;
	static const uint X68kKey_9				= 0x0a;
	static const uint X68kKey_0				= 0x0b;
	static const uint X68kKey_minus			= 0x0c;
	static const uint X68kKey_circum		= 0x0d;
	static const uint X68kKey_backslash		= 0x0e;
	static const uint X68kKey_BS			= 0x0f;

	static const uint X68kKey_TAB			= 0x10;
	static const uint X68kKey_Q				= 0x11;
	static const uint X68kKey_W				= 0x12;
	static const uint X68kKey_E				= 0x13;
	static const uint X68kKey_R				= 0x14;
	static const uint X68kKey_T				= 0x15;
	static const uint X68kKey_Y				= 0x16;
	static const uint X68kKey_U				= 0x17;
	static const uint X68kKey_I				= 0x18;
	static const uint X68kKey_O				= 0x19;
	static const uint X68kKey_P				= 0x1a;
	static const uint X68kKey_at			= 0x1b;
	static const uint X68kKey_bracketleft	= 0x1c;
	static const uint X68kKey_enter			= 0x1d;
	static const uint X68kKey_A				= 0x1e;
	static const uint X68kKey_S				= 0x1f;

	static const uint X68kKey_D				= 0x20;
	static const uint X68kKey_F				= 0x21;
	static const uint X68kKey_G				= 0x22;
	static const uint X68kKey_H				= 0x23;
	static const uint X68kKey_J				= 0x24;
	static const uint X68kKey_K				= 0x25;
	static const uint X68kKey_L				= 0x26;
	static const uint X68kKey_semicolon		= 0x27;
	static const uint X68kKey_colon			= 0x28;
	static const uint X68kKey_bracketright	= 0x29;
	static const uint X68kKey_Z				= 0x2a;
	static const uint X68kKey_X				= 0x2b;
	static const uint X68kKey_C				= 0x2c;
	static const uint X68kKey_V				= 0x2d;
	static const uint X68kKey_B				= 0x2e;
	static const uint X68kKey_N				= 0x2f;

	static const uint X68kKey_M				= 0x30;
	static const uint X68kKey_comma			= 0x31;
	static const uint X68kKey_period		= 0x32;
	static const uint X68kKey_slash			= 0x33;
	static const uint X68kKey_underscore	= 0x34;
	static const uint X68kKey_space			= 0x35;
	static const uint X68kKey_HOME			= 0x36;
	static const uint X68kKey_DEL			= 0x37;
	static const uint X68kKey_ROLLUP		= 0x38;
	static const uint X68kKey_ROLLDOWN		= 0x39;
	static const uint X68kKey_UNDO			= 0x3a;
	static const uint X68kKey_left			= 0x3b;
	static const uint X68kKey_up			= 0x3c;
	static const uint X68kKey_right			= 0x3d;
	static const uint X68kKey_down			= 0x3e;
	static const uint X68kKey_PAD_CLR		= 0x3f;

	static const uint X68kKey_PAD_divide	= 0x40;
	static const uint X68kKey_PAD_multiply	= 0x41;
	static const uint X68kKey_PAD_minus		= 0x42;
	static const uint X68kKey_PAD_7			= 0x43;
	static const uint X68kKey_PAD_8			= 0x44;
	static const uint X68kKey_PAD_9			= 0x45;
	static const uint X68kKey_PAD_plus		= 0x46;
	static const uint X68kKey_PAD_4			= 0x47;
	static const uint X68kKey_PAD_5			= 0x48;
	static const uint X68kKey_PAD_6			= 0x49;
	static const uint X68kKey_PAD_equal		= 0x4a;
	static const uint X68kKey_PAD_1			= 0x4b;
	static const uint X68kKey_PAD_2			= 0x4c;
	static const uint X68kKey_PAD_3			= 0x4d;
	static const uint X68kKey_PAD_enter		= 0x4e;
	static const uint X68kKey_PAD_0			= 0x4f;

	static const uint X68kKey_PAD_comma		= 0x50;
	static const uint X68kKey_PAD_decimal	= 0x51;
	static const uint X68kKey_kigou			= 0x52;
	static const uint X68kKey_touroku		= 0x53;
	static const uint X68kKey_HELP			= 0x54;
	static const uint X68kKey_XF1			= 0x55;
	static const uint X68kKey_XF2			= 0x56;
	static const uint X68kKey_XF3			= 0x57;
	static const uint X68kKey_XF4			= 0x58;
	static const uint X68kKey_XF5			= 0x59;
	static const uint X68kKey_kana			= 0x5a;
	static const uint X68kKey_romaji		= 0x5b;
	static const uint X68kKey_code			= 0x5c;
	static const uint X68kKey_CAPS			= 0x5d;
	static const uint X68kKey_INS			= 0x5e;
	static const uint X68kKey_hira			= 0x5f;

	static const uint X68kKey_zenkaku		= 0x60;
	static const uint X68kKey_BREAK			= 0x61;
	static const uint X68kKey_COPY			= 0x62;
	static const uint X68kKey_F1			= 0x63;
	static const uint X68kKey_F2			= 0x64;
	static const uint X68kKey_F3			= 0x65;
	static const uint X68kKey_F4			= 0x66;
	static const uint X68kKey_F5			= 0x67;
	static const uint X68kKey_F6			= 0x68;
	static const uint X68kKey_F7			= 0x69;
	static const uint X68kKey_F8			= 0x6a;
	static const uint X68kKey_F9			= 0x6b;
	static const uint X68kKey_F10			= 0x6c;

	static const uint X68kKey_SHIFT			= 0x70;
	static const uint X68kKey_CTRL			= 0x71;
	static const uint X68kKey_OPT1			= 0x72;
	static const uint X68kKey_OPT2			= 0x73;
};
